# Copyright (C) 2023 Thomas Hoppe (h0bB1T). All rights reserved.
#
# Unauthorized copying of this file via any medium is strictly prohibited.
# Proprietary and confidential.

import bpy

from typing import List

from bpy.types import Menu, Panel

from ..preferences import PreferencesPanel
from ..properties import Properties, AssetSubsetProperty
from ..registries.resource_lists_registry import ResourceListsRegistry
from ..operators.import_3d_object import ASSET_OT_import_3d_object
from ..operators.refresh_res_list import ASSET_OT_refresh_res_lists


class VIEW3D_PT_awp_assets_panel(Panel):
    """
    Sub menu to nodes, listing a subset of shaders to add to tree.
    """
    bl_idname = 'VIEW3D_PT_awp_assets_panel'
    bl_label = 'Asset Wizard Pro'
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'HEADER'
    #bl_options = {'INSTANCED'}


    def draw(self, context: bpy.types.Context):
        # This is transferred using context pointer, contains category info.
        type = PreferencesPanel.get().view3d_category_mode
        info = context.awp_as_subset # type: AssetSubsetProperty

        entries = []
        if type == 'CATALOG': entries = ResourceListsRegistry.get().col_or_objs_by_catalog(info.id)
        elif type == 'CATEGORY': entries = ResourceListsRegistry.get().col_or_objs_by_category(info.id)
        elif type == 'TOC': entries = ResourceListsRegistry.get().col_or_objs_by_toc(info.id)

        width, cols, scl = 50, 8, 5 # TODO: Make configurable.
 
        # Draw insert operator for every matching item.
        self.layout.ui_units_x = width
        g = self.layout.grid_flow(row_major=True, columns=cols, even_columns=True, even_rows=True)
        for tpl in entries:
            asset = ResourceListsRegistry.get().asset_by_uuid(tpl[0])
            c = g.column(align=True)
            c.template_icon(ResourceListsRegistry.get().asset_preview(asset), scale=scl)

            r = c.row(align=True)
            s = r.split(factor=0.5, align=True)
            is_collection = asset.type == 'Collection'
            ASSET_OT_import_3d_object.create_ui(s, asset, True, is_collection)
            ASSET_OT_import_3d_object.create_ui(s, asset, False, is_collection)


class VIEW3D_MT_awp_assets_import_menu(Menu):
    """
    The asset insert pie menu in 3D view ('Ctrl-E').
    """
    bl_idname = 'VIEW3D_MT_awp_assets_import_menu'
    bl_label = 'Asset Wizard Pro'


    @classmethod
    def poll(cls, context: bpy.types.Context):
        return context.space_data.type == 'VIEW_3D' and context.mode == 'OBJECT'
    

    def draw(self, context: bpy.types.Context):
        l = self.layout

        l.ui_units_y = 20

        type = PreferencesPanel.get().view3d_category_mode
        props = Properties.get()

        # Get list of categories depending on type from preferences.
        entries = {
            'CATALOG': props.col_or_objs_by_catalog,
            'CATEGORY': props.col_or_objs_by_category,
            'TOC': props.col_or_objs_by_toc,
        }.get(type, []) # type: List[AssetSubsetProperty]

        # Create menu entry for each item.
        g = l.column(align=True) #l.grid_flow(row_major=True, columns=0, even_columns=True, align=True)
        if entries:
            for entry in entries:
                r = g.row(align=True)
                r.context_pointer_set('awp_as_subset', entry)
                r.popover(VIEW3D_PT_awp_assets_panel.bl_idname, text=entry.text)
            g.separator()
        else:
            g.label(text='No assets found in cache, consider rescan')

        ASSET_OT_refresh_res_lists.create_ui(g, 'Refresh Cache')

