# Copyright (C) 2022 Thomas Hoppe (h0bB1T). All rights reserved.
#
# Unauthorized copying of this file via any medium is strictly prohibited.
# Proprietary and confidential.

from typing import List

from ..constants import ResourceType

class SelectableResource:
    def __init__(self, r: ResourceType):
        self.selected = False
        self.resource = r


    def toggle(self):
        self.selected = not self.selected


class ResourceSelectionRegistry:
    """
    Used to track selection of items in a list of exportable resources.
    Warning, just a single selection list is available at one time (singleton).
    """

    def __init__(self):
        self.__resources = [] # type: List[SelectableResource]


    def load(self, rscs: List[ResourceType]):
        """
        Prepare with given list of candidates.
        """
        self.__resources.clear()
        for r in rscs:
            self.__resources.append(SelectableResource(r))


    def toggle(self, resource_name: str):
        """
        Toggle selection of specified item.
        """
        for rs in self.__resources:
            if resource_name == rs.resource.name:
                rs.toggle()
                break


    def all(self) -> List[SelectableResource]:
        """
        Return all resources including flag.
        """
        return self.__resources


    def selected(self) -> List[ResourceType]:
        """
        Return selected resources.
        """
        return [ r.resource for r in self.__resources if r.selected ]


    @staticmethod
    def get() -> 'ResourceSelectionRegistry':
        global _resource_selection_instance
        return _resource_selection_instance


_resource_selection_instance = ResourceSelectionRegistry()  
