# Copyright (C) 2022 Thomas Hoppe (h0bB1T). All rights reserved.
#
# Unauthorized copying of this file via any medium is strictly prohibited.
# Proprietary and confidential.

bl_info = {
    'name' : 'Asset Wizard Pro',
    "version": (1, 6, 3),
    'author' : 'h0bB1T',
    'description' : 'Smart Assets Tools',
    'blender' : (3, 0, 0),
    'location' : 'View3D|NodeEditor > Panel',
    'category' : 'Export',
    'doc_url' : 'https://wp.h0bb1t.de/index.php/asset-wizard-pro-manual/',
}

import bpy, os

from bpy.app.handlers import persistent

from .constants import config_dir, dev_config
from .registries.config_registry import ConfigRegistry
from .registries.key_registry import KeyRegistry, KeyEntry
from .registries.icons_registry import IconsRegistry
from .registries.previews_registry import PreviewsRegistry
from .registries.resource_lists_registry import ResourceListsRegistry
from .utils.io import read_json
from .utils.tools import Measure

from .preferences import PreferencesPanel
from .properties import Properties, PropertySection, TagProperty, AssetSubsetProperty

from .headers.asset_browser_header import add_asset_browser_header, remove_asset_browser_header
from .headers.view3d_header import set_view3d_header, remove_view3d_header
from .menus.register import register_menus
from .operators.register import register_operators
from .panels.register import register_panels

from .utils.type_properties import init_type_properties, dispose_type_properties

ops = [
    TagProperty,
    AssetSubsetProperty,
    PropertySection,
    Properties,
    PreferencesPanel,

    *register_menus,
    *register_operators,
    *register_panels,
]

@persistent
def scene_update_handler(_):
    Properties.get().init()
 

def register():
    with Measure('Initialize Asset Wizard Pro'):
        # Register and setup Blender stuff.
        for op in ops:
            bpy.utils.register_class(op)

        # Create config dir.
        os.makedirs(config_dir, exist_ok=True)

        IconsRegistry.instance()

        # Eventually create default config and load current.
        ConfigRegistry.get().init_default()
        ConfigRegistry.get().read()

        # Eventually load config in dev env.
        if os.path.exists(dev_config):
            js = read_json(dev_config)
            PreferencesPanel.get().author = js.get('author', '')
            PreferencesPanel.get().auto_place = js.get('auto_place', False)
            PreferencesPanel.get().render_engine = js.get('engine', 'CYCLES')
            PreferencesPanel.get().cycles_samples = js.get('samples', 20)
            PreferencesPanel.get().dimension = js.get('dimension', 256)

        # Initialize several things.
        Properties.initialize()
        PreferencesPanel.get().transfer_defaults()

        # Add menu to Asset Browser header.
        add_asset_browser_header()
        set_view3d_header(PreferencesPanel.get().experimental_inline_import)

        # Add handler which is called on 'New Scene' to update some Properties,
        # which were cleared in this action.
        bpy.app.handlers.load_post.append(scene_update_handler)

        # Force initial load.
        ResourceListsRegistry.get()

        init_type_properties()

        # Bound keys to pies.
        KeyRegistry.instance().add_key(
            'Pie Menus', 
            'Node Editor', 
            'NODE_EDITOR', 
            [ KeyEntry('awp.invoke_pie', [], 'D') ]
        )
        KeyRegistry.instance().add_key(
            'Placer', 
            'Object Mode', 
            'EMPTY',  
            [ KeyEntry('awp.multi_purpose', [ ( 'mode', 'invoke_place' ), ], 'E') ]
        )    
        KeyRegistry.instance().add_key(
            'Replacer', 
            'Object Mode', 
            'EMPTY',  
            [ KeyEntry('awp.multi_purpose', [ ( 'mode', 'invoke_replace' ), ], 'E', shift=True) ]
        )    
        KeyRegistry.instance().add_key(
            'Asset Insert Pie Key',
            'Object Mode', 
            'EMPTY', 
            [ KeyEntry('awp.multi_purpose', [ ( 'mode', 'invoke_import_3d' ), ], 'E', ctrl=True) ]
        )


def unregister():
    with Measure('Dispose Asset Wizard Pro'):
        KeyRegistry.instance().dispose()

        bpy.app.handlers.load_post.remove(scene_update_handler)

        # Cleanup addon.
        dispose_type_properties()

        remove_view3d_header()
        remove_asset_browser_header()

        Properties.dispose()

        PreviewsRegistry.instance().dispose()

        for op in reversed(ops):
            bpy.utils.unregister_class(op) 


if __name__ == "__main__":
    register()    
