# Copyright (C) 2022 Thomas Hoppe (h0bB1T). All rights reserved.
#
# Unauthorized copying of this file via any medium is strictly prohibited.
# Proprietary and confidential.

bl_info = {
    "name" : "Shader Node Wizard",
    "version": (1, 6, 2),
    "author" : "h0bB1T",
    "description" : "Utils for Material Shader Editor",
    "blender" : (2, 93, 0),
    "location" : "Material Node Editor",
    "category" : "Material"
}

import bpy, os
from .constants import CONFIG_DIR
from .preferences import DirPathProperty, PreferencesPanel
from .properties import BrushMappingProperties, GrungeMappingProperties, MappingProperties, PBRMappingProperties, ImageMappingProperties, Properties

from .utils.tools import Measure
from .utils.node_visualizer import initialize_node_visualizer, deinitialize_node_visualizer

from .registries.previews_registry import PreviewsRegistry
from .registries.icons_registry import IconsRegistry
from .registries.key_registry import KeyEntry, KeyRegistry
from .registries.collections import Collections
from .registries.texture_registry import TextureRegistry

from .header.node_editor_header import add_node_editor_header, remove_node_editor_header

from .header.register import register_menus2
from .menus.register import register_menus
from .operators.register import register_operators
from .panels.register import register_panels

from .operators.support_invoke_pie import UI_OT_support_invoke_pie
from .operators.edit_node import UI_OT_edit_node
from .menus.view3d_paint_pie_menu import VIEW3D_MT_snw_paint_pie_menu
from .menus.view3d_edit_mode_pie_menu import VIEW3D_MT_snw_edit_mode_pie_menu
from .menus.view3d_object_mode_pie_menu import VIEW3D_MT_snw_object_mode_pie_menu


ops = [
    DirPathProperty,
    PreferencesPanel,

    MappingProperties,
    PBRMappingProperties,
    GrungeMappingProperties,
    BrushMappingProperties,
    ImageMappingProperties,
    Properties,

    *register_menus2,
    *register_menus,
    *register_operators,
    *register_panels,
]


def register():
    with Measure('Initialize Shader Node Wizard'):
        # Prepare config dir in home folder.
        os.makedirs(CONFIG_DIR, exist_ok=True)

        for op in ops:
            bpy.utils.register_class(op)

        bpy.types.Node.snw_node_info = bpy.props.StringProperty(name="SNW Info", description="Shader Node Wizard shadow information")
        bpy.types.NodeTree.snw_node_info = bpy.props.StringProperty(name="SNW Info", description="Shader Node Wizard shadow information")

        initialize_node_visualizer()

        IconsRegistry.instance()
        Properties.initialize()
        Collections.instance().init()
        TextureRegistry.instance().init()

        # Only during development.
        PreferencesPanel.restore_dev()

        # Load/scan textures.
        for path, prefix, style in PreferencesPanel.get_texture_dirs():
            TextureRegistry.instance().scan_repository(path, prefix, style)

        add_node_editor_header()

        KeyRegistry.instance().add_key(
            'Node Editor', 
            'NODE_EDITOR', [
                KeyEntry('Node Pie Key', UI_OT_support_invoke_pie.bl_idname, [], 'D'),
                KeyEntry('Node Edit Key', UI_OT_edit_node.bl_idname, [], 'E'),
            ]
        )
        
        KeyRegistry.instance().add_key(
            'Image Paint', 
            'EMPTY',  [
                KeyEntry('Paint Pie Key', 'wm.call_menu_pie', [ ('name', VIEW3D_MT_snw_paint_pie_menu.bl_idname) ], 'D')
            ]
        )

        KeyRegistry.instance().add_key(
            'Sculpt', 
            'EMPTY',  [
                KeyEntry('Sculpt Pie Key', 'wm.call_menu_pie', [ ('name', VIEW3D_MT_snw_paint_pie_menu.bl_idname) ], 'D')
            ]
        )        

        KeyRegistry.instance().add_key(
            'Mesh', 
            'EMPTY',  [
                KeyEntry('3D Edit Pie Key', 'wm.call_menu_pie', [ ('name', VIEW3D_MT_snw_edit_mode_pie_menu.bl_idname) ], 'D')
            ]
        )    

        KeyRegistry.instance().add_key(
            'Object Mode', 
            'EMPTY',  [
                KeyEntry('3D Edit Pie Key', 'wm.call_menu_pie', [ ('name', VIEW3D_MT_snw_object_mode_pie_menu.bl_idname) ], 'D')
            ]
        )      

 
def unregister():
    with Measure('Dispose Shader Node Wizard'):
        KeyRegistry.instance().dispose()

        remove_node_editor_header()

        Collections.instance().dispose()
        Properties.dispose()
        TextureRegistry.instance().dispose()

        PreviewsRegistry.instance().dispose()

        deinitialize_node_visualizer()

        del bpy.types.NodeTree.snw_node_info
        del bpy.types.Node.snw_node_info

        for op in reversed(ops):
            bpy.utils.unregister_class(op) 


if __name__ == "__main__":
    register()    
