# Copyright (C) 2022 Thomas Hoppe (h0bB1T). All rights reserved.
#
# Unauthorized copying of this file via any medium is strictly prohibited.
# Proprietary and confidential.

from bpy.types import Menu

from ..properties import Properties
from ..preferences import PreferencesPanel

from ..utils.ui import insert_pbr_channels

from ..registries.texture_registry import RepositoryType, TextureRegistry
from ..snw.ui import insert_preview_update

from ..operators.support_cascade_pie_menu import UI_OT_support_cascade_pie_menu


class NODE_MT_snw_pbr_category_pie_menu(Menu):
    """
    Pie Menu to select PBR category.
    """
    bl_idname = "NODE_MT_snw_pbr_category_pie_menu"
    bl_label = "Select PBR Category"


    @classmethod
    def poll(cls, context):
        return context.space_data.type == 'NODE_EDITOR' and \
            context.space_data.node_tree is not None  and \
            context.space_data.tree_type == 'ShaderNodeTree'    


    def draw(self, context):
        pie = self.layout.menu_pie()

        pie.separator()
        c = pie.column().grid_flow(columns=4, align=True)
        for t in TextureRegistry.instance().get_category_enums(RepositoryType.PBR):
            op = c.operator(UI_OT_support_cascade_pie_menu.bl_idname, text=t[1])
            op.mode = 'PBR'
            op.param = t[0]


class NODE_MT_snw_pbr_set_pie_menu(Menu):
    """
    Displays all textures from a PBR category and creates the PBR material.
    """
    bl_idname = "NODE_MT_snw_pbr_set_pie_menu"
    bl_label = "Select PBR Set"

    @classmethod
    def poll(cls, context):
        return context.space_data.type == 'NODE_EDITOR' and \
            context.space_data.node_tree is not None  and \
            context.space_data.tree_type == 'ShaderNodeTree'    


    def draw(self, context):
        pie = self.layout.menu_pie()
        pie.separator()

        props = Properties.get().pbr
        p = pie.column()
        entries = TextureRegistry.instance().get_image_list(RepositoryType.PBR, props.categories)
        if entries: 
            insert_preview_update(p.row(align=True), RepositoryType.PBR, props.categories)
            c = p.box().grid_flow(columns=PreferencesPanel.get().preview_columns_pie, row_major=True, align=True)
            for i in entries:
                b = c.box().column()
                b.template_icon(icon_value=i.preview_icon_id, scale=PreferencesPanel.get().preview_scale_pie)
                insert_pbr_channels(b.row(align=True), i, short=True)
                if i.variant_count() == 1:
                    op = b.operator('snw.create_pbr_node', text=i.info.pbr.variant_name, icon='NODE_MATERIAL')
                    op.category = props.categories
                    op.pbr = i.key
                    op.variant = 0
                    op.bin_hash = i.get_bin_hash()
                else:
                    r = b.row(align=True)
                    for n, v in enumerate([i.info.pbr, *i.info.pbr.variants]):
                        op = r.operator('snw.create_pbr_node', text=v.variant_name, icon='NODE_MATERIAL')
                        op.category = props.categories
                        op.pbr = i.key
                        op.variant = n
                        op.bin_hash = i.get_bin_hash(n)

        else:
            p.box().label(text='No content')
