# Copyright (C) 2022 Thomas Hoppe (h0bB1T). All rights reserved.
#
# Unauthorized copying of this file via any medium is strictly prohibited.
# Proprietary and confidential.

import bpy

from bpy.types import Operator
from bpy.props import IntProperty, StringProperty, BoolProperty, FloatVectorProperty

from ..utils.blender import drop_node
from ..operators.helpers.drop_node import DropNodeOperator
from ..operators.helpers.setup_image import create_color_image, create_effect_image, setup_color_image, setup_effect_image


class UI_OT_create_node_new_image(DropNodeOperator, Operator):
    """
    Create new, empty image from PIE menu with selected image dimensions.
    """
    bl_idname = 'snw.create_node_new_image'
    bl_label = 'Create Paint Image'
    bl_description = 'Create new image with the given resolution plus image node'
    bl_options = {'REGISTER'}

    resolution: IntProperty() # type: ignore
    mode: StringProperty() # type: ignore
    uv_map: StringProperty() # type: ignore
    use_alpha: BoolProperty() # type: ignore
    use_32bit: BoolProperty() # type: ignore
    fill: FloatVectorProperty(size=4, min=0.0, max=1.0) # type: ignore


    @classmethod
    def description(cls, context, properties):
        return {
            'FX': f'Create gray level effect image, with {properties.resolution}x{properties.resolution}',
            'IM': f'Create diffuse image, with {properties.resolution}x{properties.resolution}',
        }.get(properties.mode, '??')


    def execute(self, context: bpy.types.Context):
        im = bpy.data.images.new(
            f'{self.mode}_{self.resolution}', 
            self.resolution, 
            self.resolution,
            alpha=self.use_alpha,
            float_buffer=self.use_32bit
        )
        im.generated_color = self.fill

        if self.mode == 'FX':
            nb = create_effect_image(im.name)
            setup_effect_image(nb, im, self.uv_map)
            node = nb.tree
        elif self.mode == 'IM':
            nb = create_color_image(im.name)
            setup_color_image(nb, im, self.uv_map)
            node = nb.tree

        return drop_node(node)
