# Copyright (C) 2022 Thomas Hoppe (h0bB1T). All rights reserved.
#
# Unauthorized copying of this file via any medium is strictly prohibited.
# Proprietary and confidential.

import bpy

from .simple import setup_gray_texture
from .setup_grunge import create_grunge
from ...snw.node_info import encode_image_paint_info
from ...utils.node_builder_cycles import NodeBuilderCycles


def create_effect_image(name: str) -> NodeBuilderCycles:
    """
    Identical for now, so just wrap the call.
    """
    return create_grunge(name, False, default_bump=20.0, max_bump=100.0)


def setup_effect_image(nb: NodeBuilderCycles, image: bpy.types.Image, uv_map: str):
    """
    Build node tree for paintable images that act like grunge maps.
    """

    nb.tree.name = image.name
    nb.tree.snw_node_info = encode_image_paint_info('', 'FX')

    uv = nb.add_uv_map(uv_map).outputs[0]
    proj = 'FLAT'
    blend = 0.5

    tex = nb.add_image(
        image,
        colorspace='Non-Color',
        projection=proj,
        blend=blend,
        vector=uv
    ).outputs['Color']

    setup_gray_texture(nb, tex, image.size)


def create_color_image(name: str) -> NodeBuilderCycles:
    """
    Create an empty image node to hold an image for use as diffuse/color.
    """
    nb = NodeBuilderCycles()
    nb.build_node_tree(name)

    nb.add_inputs(
        [
            ( 'FloatFactor', 'Hue', 0.5, 0.0, 1.0 ),
            ( 'FloatFactor', 'Saturation', 1.0, 0.0, 2.0 ),
            ( 'FloatFactor', 'Value', 1.0, 0.0, 2.0 ),
            ( 'FloatFactor', 'Brightness', 0.0, -0.5, 0.5 ),
            ( 'FloatFactor', 'Contrast', 0.0, -0.5, 0.5 ),
            ( 'FloatFactor', 'Alpha Offset', 0.02, 0.0, 0.1 ),
            ( 'FloatFactor', 'Alpha Exponent', 0.01, 0.0, 0.1 ),
        ]
    )

    nb.add_outputs([
        ( 'Color' , 'Color' ),
        ( 'Float', 'Alpha' ),
        ( 'Float', 'Alpha from Color' ),
    ])

    return nb    


def setup_color_image(nb: NodeBuilderCycles, image: bpy.types.Image, uv_map: str):
    """
    Build node tree for paintable images that act like grunge maps.
    """

    nb.tree.name = image.name
    nb.tree.snw_node_info = encode_image_paint_info('', 'IM')

    uv = nb.add_uv_map(uv_map).outputs[0]
    proj = 'FLAT'
    blend = 0.5

    tex = nb.add_image(
        image,
        colorspace='sRGB',
        projection=proj,
        blend=blend,
        vector=uv
    )

    color = nb.add_brightness_contrast(
        brightness=nb.input('Brightness'),
        contrast=nb.input('Contrast'),
        color=nb.add_hsv(
            hue=nb.input('Hue'),
            saturation=nb.input('Saturation'),
            value=nb.input('Value'),
            color=tex.outputs['Color']
        ).outputs[0]
    ).outputs[0]

    nb.wire_sockets(color, nb.output('Color'))
    nb.wire_sockets(tex.outputs['Alpha'], nb.output('Alpha'))
    nb.wire_sockets(
        nb.add_math_power(
            nb.add_map_range(
                value=tex.outputs['Color'],
                fmin=nb.input('Alpha Offset')
            ).outputs[0],
            nb.input('Alpha Exponent')
        ),
        nb.output('Alpha from Color')
    )

    nb.arrange()

