# Copyright (C) 2022 Thomas Hoppe (h0bB1T). All rights reserved.
#
# Unauthorized copying of this file via any medium is strictly prohibited.
# Proprietary and confidential.

from .simple import setup_simple_image_texture
from ...utils.dev import dbg
from ...utils.node_builder_cycles import NodeBuilderCycles
from ...registries.texture_registry import RepositoryType, TextureRegistry
from ...registries.textures.builders.image_builder import build_image_node_setup
from ...snw.node_info import encode_image_info


def _enable_uv_offset(nb: NodeBuilderCycles):
    name = 'UV Offset'
    if not nb.tree_has_input(name):
        nb.add_inputs([ ( 'Vector', name ), ], after=[ 'UV' ]) 


def create_image_texture(name: str, add_uv: bool) -> NodeBuilderCycles:
    """
    Create an empty image texture node.
    """
    nb = NodeBuilderCycles()
    nb.build_node_tree(name)

    if add_uv:
        nb.add_inputs(
            [
                ( 'Vector' , 'UV' ),
                ( 'Vector', 'UV Offset' ),
                ( 'Float' , 'Scale', 1.0 ),
            ],
            hidden={ 'UV' }
        )

    nb.add_inputs(
        [
            ( 'Float', 'Hue', 0.5 ),
            ( 'Float', 'Saturation', 1.0 ),
            ( 'Float', 'Value', 1.0 ),
        ]
    )

    nb.add_outputs([
        ( 'Color' , 'Color (sRGB)' ),
        ( 'Color' , 'Color (Non-Color)' ),
        ( 'Float', 'Alpha' ),
    ])

    return nb


def setup_image_texture(
    nb: NodeBuilderCycles,
    category: str,
    image: str,
    bin_hash: str = '', 
    mapping: str = 'UV',
    blend: float = 0.5,
    interpolation: str = 'Linear',
    anti_repeat_style: str = 'OFF',
    anti_repeat_scale: float = 2,
    anti_repeat_distortion: float = 0.1,
    anti_repeat_style_param_0: float = 0.1,
    anti_repeat_seed: float = 0
    ):
    """
    Full a (newly created or previously cleared) grunge node with content.
    """

    dbg(f'Update Image Texture: {category}/{image} with {mapping}/{interpolation}')
    imagen = TextureRegistry.instance().get_image(RepositoryType.Image, category, image)

    nb.tree.name = imagen.name
    nb.tree.snw_node_info = encode_image_info(
        '', 
        category, 
        image, 
        bin_hash, 
        mapping,
        blend,
        interpolation, 
        anti_repeat_style,
        anti_repeat_scale,
        anti_repeat_distortion,
        anti_repeat_style_param_0,
        anti_repeat_seed
    )

    if mapping == 'UV':
        uv = nb.add_auto_uv(nb.input('UV'), nb.input('Scale'))
    elif mapping == 'LBOX':
        uv = nb.add_local_box_uv_mapping(nb.input('Scale'))
    elif mapping == 'GENBOX':
        uv = nb.add_generate_box_uv_mapping(nb.input('Scale'))        
    elif mapping == 'GBOX':
        uv = nb.add_global_box_uv_mapping(nb.input('Scale'))
    else:
        raise Exception('Unknown UV Mapping')
    
    _enable_uv_offset(nb) # For older nodes.
    uv = nb.add_vector_math_add(uv, nb.input('UV Offset'))    

    if anti_repeat_style != 'OFF':
        uv = nb.add_uv_anti_repeat(
            uv,
            anti_repeat_style,
            anti_repeat_scale,
            anti_repeat_distortion,
            anti_repeat_style_param_0,
            anti_repeat_seed
        )        

    proj = 'FLAT' if mapping == 'UV' else 'BOX'

    color_socket, alpha_socket, image = build_image_node_setup(nb, imagen.info.image, uv, proj, blend, interpolation)
    setup_simple_image_texture(nb, color_socket, alpha_socket)
    