# Copyright (C) 2022 Thomas Hoppe (h0bB1T). All rights reserved.
#
# Unauthorized copying of this file via any medium is strictly prohibited.
# Proprietary and confidential.


from typing import Dict, Tuple
from uuid import uuid4


class EnumRegistry:
    """
    Used to support non-latin id's in EnumProperties. Using user
    supplied strings (e.g. file names) as id may fail, as not all chars are supported here.
    This replaces the id with a uuid and offers function to interact without much
    more code on use side. 
    Usage: Rename the property to PROP_back and add a getter/setter property.
    Warning: Just use for properties that were not saved into .blend files, the uuid's
    are temporary and were different every time blender is started!
    """

    def __init__(self):
        self.__i2s = {} # type: Dict[str, str]
        self.__s2i = {} # type: Dict[str, str]
        self.__enum3 = {} # type: Dict[str, Tuple[str, str, str]]


    def __replace_for_enum3(self, id_string: str, text: str, description: str) -> Tuple[str, str, str]:
        if id_string not in self.__s2i:
            id = str(uuid4())
            self.__s2i[id_string] = id
            self.__i2s[id] = id_string
            self.__enum3[id] = (id, text, description)
        else:
            id = self.__s2i[id_string]

        return self.__enum3[id]


    def __id_for_str(self, string: str) -> str:
        if string not in self.__s2i:
            id = str(uuid4())
            self.__s2i[string] = id
            self.__i2s[id] = string
            
        return self.__s2i[string]


    def __str_for_id(self, id: str) -> str:
        return self.__i2s.get(id, '??')


    @staticmethod
    def replace_for_enum3(id_string: str, text: str, description: str) -> Tuple[str, str, str]:
        global _enum_registry_instance
        return _enum_registry_instance.__replace_for_enum3(id_string, text, description)


    @staticmethod
    def id_for_str(string: str) -> str:
        global _enum_registry_instance
        return _enum_registry_instance.__id_for_str(string)


    @staticmethod
    def str_for_id(id: str) -> str:
        global _enum_registry_instance
        return _enum_registry_instance.__str_for_id(id)


_enum_registry_instance = EnumRegistry()
