# Copyright (C) 2022 Thomas Hoppe (h0bB1T). All rights reserved.
#
# Unauthorized copying of this file via any medium is strictly prohibited.
# Proprietary and confidential.

from dataclasses import dataclass, field
from typing import Dict, List


@dataclass
class PBREntry:
    name: str
    variant_name: str = None
    preview: str = None

    diffuse: str = None
    alpha: str = None
    ao: str = None
    specular: str = None
    metal: str = None
    roughness: str = None
    glossiness: str = None
    normal: str = None
    height: str = None
    displace: str = None
    emission: str = None
    arm: str = None # PolyHaven -> AO, Roughness, Metal mixed in rgb

    variants: List['PBREntry'] = field(default_factory=list)

    def has_ao(self) -> bool: return self.ao != None or self.arm != None
    def has_specular(self) -> bool: return self.specular != None
    def has_metallic(self) -> bool: return self.metal != None or self.arm != None
    def has_displace(self) -> bool: return self.displace != None
    def has_emission(self) -> bool: return self.emission != None
    

    @staticmethod
    def load(data: Dict) -> 'PBREntry':
        return PBREntry(
            data.get('name', 'UNKNOWN'),
            data.get('variant_name'),
            data.get('preview'),
            data.get('diffuse'),
            data.get('alpha'),
            data.get('ao'),
            data.get('specular'),
            data.get('metal'),
            data.get('roughness'),
            data.get('glossiness'),
            data.get('normal'),
            data.get('height'),
            data.get('displace'),
            data.get('emission'),
            data.get('arm'),
            [ PBREntry.load(e) for e in data.get('variants', list()) ]
        )

    def save(self) -> Dict:
        return {
            'name': self.name,
            'variant_name': self.variant_name,
            'preview': self.preview,
            'diffuse': self.diffuse,
            'alpha': self.alpha,
            'ao': self.ao,
            'specular': self.specular,
            'metal': self.metal,
            'roughness': self.roughness,
            'glossiness': self.glossiness,
            'normal': self.normal,
            'height': self.height,
            'displace': self.displace,
            'emission': self.emission,
            'arm': self.arm,
            'variants': [ e.save() for e in self.variants ]
        }
    